# Configuration Guide

This guide covers all configuration options available in the Email Sender application.

## Configuration File Overview

The main configuration is stored in `config.php`. This file contains all SMTP settings, security options, and application preferences.

## SMTP Configuration

### Basic SMTP Settings

```php
// SMTP Server Settings
define('SMTP_HOST', 'smtp.example.com');     // SMTP server hostname
define('SMTP_PORT', 587);                    // SMTP port number
define('SMTP_ENCRYPTION', 'tls');            // Encryption: 'tls', 'ssl', or ''
define('SMTP_USERNAME', 'user@example.com'); // SMTP username
define('SMTP_PASSWORD', 'password');         // SMTP password
```

### Port Selection Guide

| Port | Encryption | Description |
|------|------------|-------------|
| 25   | None       | Standard SMTP (often blocked by ISPs) |
| 587  | TLS        | Submission port (recommended) |
| 465  | SSL        | Legacy secure SMTP |
| 2525 | TLS        | Alternative submission port |

### Encryption Options

- **TLS (Transport Layer Security)**: Modern, secure encryption
- **SSL (Secure Sockets Layer)**: Legacy encryption (still supported)
- **None**: No encryption (not recommended for production)

## Provider-Specific Configurations

### Gmail Setup

```php
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@gmail.com');
define('SMTP_PASSWORD', 'your-app-password');
```

**Gmail Requirements:**
1. Enable 2-Factor Authentication
2. Generate App Password:
   - Go to Google Account settings
   - Security → 2-Step Verification → App passwords
   - Generate password for "Mail"
   - Use this 16-digit password in configuration

### Microsoft Outlook/Office 365

```php
define('SMTP_HOST', 'smtp-mail.outlook.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@outlook.com');
define('SMTP_PASSWORD', 'your-password');
```

**For Office 365 Business:**
```php
define('SMTP_HOST', 'smtp.office365.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
```

### Yahoo Mail

```php
define('SMTP_HOST', 'smtp.mail.yahoo.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@yahoo.com');
define('SMTP_PASSWORD', 'your-app-password');
```

**Yahoo Requirements:**
1. Enable 2-Step Verification
2. Generate App Password in Account Security settings

### Custom/Business Email

```php
define('SMTP_HOST', 'mail.yourdomain.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-email@yourdomain.com');
define('SMTP_PASSWORD', 'your-password');
```

**Common Business Email Ports:**
- **cPanel/WHM**: Usually port 587 with TLS
- **Plesk**: Typically port 587 or 465
- **Exchange**: Port 587 with TLS

## Sender Configuration

### Default Sender Information

```php
define('DEFAULT_FROM_EMAIL', 'noreply@yourdomain.com');
define('DEFAULT_FROM_NAME', 'Your Application Name');
```

**Best Practices:**
- Use a no-reply email address for automated emails
- Choose a recognizable sender name
- Ensure the from email matches your SMTP authentication

### Dynamic Sender Names

The application allows users to specify a custom "From Name" in the form, which will override the default setting for that specific email.

## Security Configuration

### CORS (Cross-Origin Resource Sharing)

```php
// Allow all origins (development only)
define('ALLOWED_ORIGINS', '*');

// Allow specific domains (production)
define('ALLOWED_ORIGINS', 'https://yourdomain.com,https://www.yourdomain.com');

// Allow single domain
define('ALLOWED_ORIGINS', 'https://yourdomain.com');
```

### Debug Mode

```php
// Enable detailed error logging (development)
define('DEBUG_MODE', true);

// Disable error details (production)
define('DEBUG_MODE', false);
```

**Debug Mode Effects:**
- **Enabled**: Detailed error messages, PHP error logging
- **Disabled**: Generic error messages, minimal logging

## Advanced Configuration

### Custom Error Messages

You can customize error messages by modifying the `send_email.php` file:

```php
// Custom validation messages
$errors = [];

if (empty($to)) {
    $errors[] = 'Please provide a recipient email address';
}

if (!validateEmail($to)) {
    $errors[] = 'The email address format is invalid';
}
```

### Rate Limiting (Optional)

To implement basic rate limiting, add this to `send_email.php`:

```php
// Simple rate limiting (requires session support)
session_start();

$max_emails_per_hour = 10;
$current_time = time();
$hour_ago = $current_time - 3600;

if (!isset($_SESSION['email_timestamps'])) {
    $_SESSION['email_timestamps'] = [];
}

// Remove old timestamps
$_SESSION['email_timestamps'] = array_filter(
    $_SESSION['email_timestamps'],
    function($timestamp) use ($hour_ago) {
        return $timestamp > $hour_ago;
    }
);

// Check rate limit
if (count($_SESSION['email_timestamps']) >= $max_emails_per_hour) {
    echo json_encode([
        'success' => false,
        'message' => 'Rate limit exceeded. Please try again later.'
    ]);
    exit();
}

// Add current timestamp
$_SESSION['email_timestamps'][] = $current_time;
```

### Custom Validation Rules

Modify validation in `send_email.php`:

```php
// Custom subject length
if (strlen($subject) < 5 || strlen($subject) > 100) {
    $errors[] = 'Subject must be between 5 and 100 characters';
}

// Custom message length
if (strlen($message) < 20 || strlen($message) > 5000) {
    $errors[] = 'Message must be between 20 and 5000 characters';
}

// Blocked domains
$blocked_domains = ['tempmail.com', 'guerrillamail.com'];
$email_domain = substr(strrchr($to, "@"), 1);
if (in_array($email_domain, $blocked_domains)) {
    $errors[] = 'This email domain is not allowed';
}
```

## Environment-Specific Configurations

### Development Environment

```php
// config.php for development
define('SMTP_HOST', 'smtp.mailtrap.io');  // Use Mailtrap for testing
define('SMTP_PORT', 2525);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'your-mailtrap-username');
define('SMTP_PASSWORD', 'your-mailtrap-password');
define('DEBUG_MODE', true);
define('ALLOWED_ORIGINS', '*');
```

### Staging Environment

```php
// config.php for staging
define('SMTP_HOST', 'smtp.gmail.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'staging@yourdomain.com');
define('SMTP_PASSWORD', 'staging-app-password');
define('DEBUG_MODE', true);
define('ALLOWED_ORIGINS', 'https://staging.yourdomain.com');
```

### Production Environment

```php
// config.php for production
define('SMTP_HOST', 'smtp.yourdomain.com');
define('SMTP_PORT', 587);
define('SMTP_ENCRYPTION', 'tls');
define('SMTP_USERNAME', 'noreply@yourdomain.com');
define('SMTP_PASSWORD', 'secure-production-password');
define('DEBUG_MODE', false);
define('ALLOWED_ORIGINS', 'https://yourdomain.com');
```

## Configuration Validation

### Testing SMTP Configuration

Create a test script to validate your SMTP settings:

```php
<?php
// test-config.php
require_once 'config.php';

echo "Testing SMTP Configuration...\n";
echo "Host: " . SMTP_HOST . "\n";
echo "Port: " . SMTP_PORT . "\n";
echo "Encryption: " . SMTP_ENCRYPTION . "\n";
echo "Username: " . SMTP_USERNAME . "\n";

// Test connection
$connection = @fsockopen(SMTP_HOST, SMTP_PORT, $errno, $errstr, 10);
if ($connection) {
    echo "✓ SMTP server is reachable\n";
    fclose($connection);
} else {
    echo "✗ Cannot reach SMTP server: $errstr\n";
}
?>
```

### Configuration Checklist

Before deploying to production:

- [ ] SMTP credentials are correct and tested
- [ ] Debug mode is disabled
- [ ] CORS origins are properly restricted
- [ ] File permissions are secure (600 for config.php)
- [ ] Default sender information is configured
- [ ] SSL/TLS encryption is enabled
- [ ] Rate limiting is implemented (if needed)
- [ ] Error handling is appropriate for environment

## Backup and Recovery

### Configuration Backup

```bash
# Backup configuration
cp config.php config.php.backup

# Backup with timestamp
cp config.php "config.php.backup.$(date +%Y%m%d_%H%M%S)"
```

### Configuration Templates

Create environment-specific templates:

```bash
# Create templates directory
mkdir config-templates

# Save environment templates
cp config.php config-templates/config.production.php
cp config.php config-templates/config.staging.php
cp config.php config-templates/config.development.php
```

## Troubleshooting Configuration Issues

### Common Configuration Problems

1. **Authentication Failed**
   - Verify username and password
   - Check if 2FA/App passwords are required
   - Ensure "Less secure app access" is enabled (if applicable)

2. **Connection Timeout**
   - Verify SMTP host and port
   - Check firewall settings
   - Try alternative ports (587, 465, 2525)

3. **TLS/SSL Errors**
   - Verify encryption setting matches server requirements
   - Check PHP OpenSSL extension
   - Try different encryption methods

4. **Permission Denied**
   - Check file permissions on config.php
   - Verify web server can read configuration
   - Ensure proper ownership settings

### Debug Configuration

Enable detailed debugging:

```php
// Add to config.php for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Enable SMTP debug output
define('SMTP_DEBUG', 2);  // Add this line
```

Then modify `send_email.php` to use debug mode:

```php
if (defined('SMTP_DEBUG')) {
    $mail->SMTPDebug = SMTP_DEBUG;
    $mail->Debugoutput = 'html';
}
```

This comprehensive configuration guide should help you set up and customize the Email Sender application for your specific needs.

