document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('emailForm');
    const sendBtn = document.getElementById('sendBtn');
    const resultMessage = document.getElementById('resultMessage');
    
    // Form validation
    function validateEmail(email) {
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return emailRegex.test(email);
    }
    
    function showError(fieldId, message) {
        const errorElement = document.getElementById(fieldId + '-error');
        const inputElement = document.getElementById(fieldId);
        
        errorElement.textContent = message;
        errorElement.classList.add('show');
        inputElement.style.borderColor = '#e74c3c';
        
        setTimeout(() => {
            errorElement.classList.remove('show');
            inputElement.style.borderColor = '#e1e5e9';
        }, 3000);
    }
    
    function clearErrors() {
        const errorElements = document.querySelectorAll('.error-message');
        const inputElements = document.querySelectorAll('input, textarea');
        
        errorElements.forEach(error => {
            error.classList.remove('show');
        });
        
        inputElements.forEach(input => {
            input.style.borderColor = '#e1e5e9';
        });
    }
    
    function validateForm() {
        clearErrors();
        let isValid = true;
        
        const to = document.getElementById('to').value.trim();
        const subject = document.getElementById('subject').value.trim();
        const message = document.getElementById('message').value.trim();
        
        if (!to) {
            showError('to', 'Recipient email is required');
            isValid = false;
        } else if (!validateEmail(to)) {
            showError('to', 'Please enter a valid email address');
            isValid = false;
        }
        
        if (!subject) {
            showError('subject', 'Subject is required');
            isValid = false;
        } else if (subject.length < 3) {
            showError('subject', 'Subject must be at least 3 characters long');
            isValid = false;
        }
        
        if (!message) {
            showError('message', 'Message is required');
            isValid = false;
        } else if (message.length < 10) {
            showError('message', 'Message must be at least 10 characters long');
            isValid = false;
        }
        
        return isValid;
    }
    
    function showResult(message, type) {
        resultMessage.textContent = message;
        resultMessage.className = `result-message ${type} show`;
        
        setTimeout(() => {
            resultMessage.classList.remove('show');
        }, 5000);
    }
    
    function setLoading(loading) {
        if (loading) {
            sendBtn.classList.add('loading');
            sendBtn.disabled = true;
        } else {
            sendBtn.classList.remove('loading');
            sendBtn.disabled = false;
        }
    }
    
    // Form submission
    form.addEventListener('submit', async function(e) {
        e.preventDefault();
        
        if (!validateForm()) {
            return;
        }
        
        setLoading(true);
        
        const formData = new FormData(form);
        
        try {
            const response = await fetch('send_email.php', {
                method: 'POST',
                body: formData
            });
            
            const result = await response.json();
            
            if (result.success) {
                showResult('Email sent successfully!', 'success');
                form.reset();
            } else {
                showResult(result.message || 'Failed to send email. Please try again.', 'error');
            }
        } catch (error) {
            console.error('Error:', error);
            showResult('Network error. Please check your connection and try again.', 'error');
        } finally {
            setLoading(false);
        }
    });
    
    // Real-time validation
    document.getElementById('to').addEventListener('blur', function() {
        const email = this.value.trim();
        if (email && !validateEmail(email)) {
            showError('to', 'Please enter a valid email address');
        }
    });
    
    document.getElementById('subject').addEventListener('blur', function() {
        const subject = this.value.trim();
        if (subject && subject.length < 3) {
            showError('subject', 'Subject must be at least 3 characters long');
        }
    });
    
    document.getElementById('message').addEventListener('blur', function() {
        const message = this.value.trim();
        if (message && message.length < 10) {
            showError('message', 'Message must be at least 10 characters long');
        }
    });
    
    // Character counter for message
    const messageTextarea = document.getElementById('message');
    const messageGroup = messageTextarea.parentElement;
    
    const charCounter = document.createElement('div');
    charCounter.style.cssText = `
        position: absolute;
        bottom: 10px;
        right: 15px;
        font-size: 0.8rem;
        color: #666;
        background: rgba(255, 255, 255, 0.9);
        padding: 2px 6px;
        border-radius: 4px;
    `;
    messageGroup.style.position = 'relative';
    messageGroup.appendChild(charCounter);
    
    messageTextarea.addEventListener('input', function() {
        const length = this.value.length;
        charCounter.textContent = `${length} characters`;
        
        if (length > 1000) {
            charCounter.style.color = '#e74c3c';
        } else {
            charCounter.style.color = '#666';
        }
    });
    
    // Initialize character counter
    charCounter.textContent = '0 characters';
});

