<?php
/**
 * Email Sender Backend
 * 
 * Handles email sending via SMTP using PHPMailer
 */

// Include configuration
require_once 'config.php';

// Set content type to JSON
header('Content-Type: application/json');

// Handle CORS
$origin = isset($_SERVER['HTTP_ORIGIN']) ? $_SERVER['HTTP_ORIGIN'] : '';
if (ALLOWED_ORIGINS === '*' || in_array($origin, explode(',', ALLOWED_ORIGINS))) {
    header('Access-Control-Allow-Origin: ' . (ALLOWED_ORIGINS === '*' ? '*' : $origin));
}
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Only allow POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit();
}

// Function to sanitize input
function sanitizeInput($input) {
    return htmlspecialchars(strip_tags(trim($input)), ENT_QUOTES, 'UTF-8');
}

// Function to validate email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL) !== false;
}

// Function to send email using PHP's built-in mail function (fallback)
function sendEmailBuiltIn($to, $subject, $message, $fromEmail, $fromName) {
    $headers = array();
    $headers[] = 'MIME-Version: 1.0';
    $headers[] = 'Content-type: text/html; charset=UTF-8';
    $headers[] = 'From: ' . $fromName . ' <' . $fromEmail . '>';
    $headers[] = 'Reply-To: ' . $fromEmail;
    $headers[] = 'X-Mailer: PHP/' . phpversion();
    
    $htmlMessage = nl2br(htmlspecialchars($message, ENT_QUOTES, 'UTF-8'));
    
    return mail($to, $subject, $htmlMessage, implode("\r\n", $headers));
}

// Function to send email using SMTP (with PHPMailer if available)
function sendEmailSMTP($to, $subject, $message, $fromEmail, $fromName) {
    // Check if PHPMailer is available
    if (class_exists('PHPMailer\PHPMailer\PHPMailer')) {
        return sendWithPHPMailer($to, $subject, $message, $fromEmail, $fromName);
    } else {
        // Fallback to built-in mail function
        return sendEmailBuiltIn($to, $subject, $message, $fromEmail, $fromName);
    }
}

// Function to send email with PHPMailer
function sendWithPHPMailer($to, $subject, $message, $fromEmail, $fromName) {
    require_once 'vendor/autoload.php';
    
    $mail = new PHPMailer\PHPMailer\PHPMailer(true);
    
    try {
        // Server settings
        $mail->isSMTP();
        $mail->Host = SMTP_HOST;
        $mail->SMTPAuth = true;
        $mail->Username = SMTP_USERNAME;
        $mail->Password = SMTP_PASSWORD;
        $mail->SMTPSecure = SMTP_ENCRYPTION;
        $mail->Port = SMTP_PORT;
        
        // Recipients
        $mail->setFrom($fromEmail, $fromName);
        $mail->addAddress($to);
        $mail->addReplyTo($fromEmail, $fromName);
        
        // Content
        $mail->isHTML(true);
        $mail->Subject = $subject;
        $mail->Body = nl2br(htmlspecialchars($message, ENT_QUOTES, 'UTF-8'));
        $mail->AltBody = strip_tags($message);
        
        $mail->send();
        return true;
    } catch (Exception $e) {
        if (DEBUG_MODE) {
            error_log("PHPMailer Error: " . $mail->ErrorInfo);
        }
        return false;
    }
}

try {
    // Get and validate input data
    $to = isset($_POST['to']) ? sanitizeInput($_POST['to']) : '';
    $subject = isset($_POST['subject']) ? sanitizeInput($_POST['subject']) : '';
    $message = isset($_POST['message']) ? sanitizeInput($_POST['message']) : '';
    $fromName = isset($_POST['from_name']) ? sanitizeInput($_POST['from_name']) : DEFAULT_FROM_NAME;
    
    // Validation
    $errors = [];
    
    if (empty($to)) {
        $errors[] = 'Recipient email is required';
    } elseif (!validateEmail($to)) {
        $errors[] = 'Invalid recipient email address';
    }
    
    if (empty($subject)) {
        $errors[] = 'Subject is required';
    } elseif (strlen($subject) < 3) {
        $errors[] = 'Subject must be at least 3 characters long';
    }
    
    if (empty($message)) {
        $errors[] = 'Message is required';
    } elseif (strlen($message) < 10) {
        $errors[] = 'Message must be at least 10 characters long';
    }
    
    if (!empty($errors)) {
        echo json_encode([
            'success' => false,
            'message' => implode(', ', $errors)
        ]);
        exit();
    }
    
    // Send email
    $emailSent = sendEmailSMTP($to, $subject, $message, DEFAULT_FROM_EMAIL, $fromName);
    
    if ($emailSent) {
        echo json_encode([
            'success' => true,
            'message' => 'Email sent successfully!'
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Failed to send email. Please check your SMTP configuration.'
        ]);
    }
    
} catch (Exception $e) {
    if (DEBUG_MODE) {
        error_log("Email sending error: " . $e->getMessage());
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred while sending the email.'
    ]);
}
?>

